/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License.
 */

"use strict";

/**
 * `InternalOpenIDError` error.
 *
 * InternalOpenIDError wraps errors generated by node-openid.  By wrapping these
 * objects, error messages can be formatted in a manner that aids in debugging
 * OpenID issues.
 *
 * @api public
 */
function InternalOpenIDError(message, err) {
  Error.call(this);
  Error.captureStackTrace(this, InternalOpenIDError);
  this.name = "InternalOpenIDError";
  this.message = message;
  this.openidError = err;
}

/**
 * Inherit from `Error`.
 */
InternalOpenIDError.prototype = Object.create(Error.prototype);

/**
 * Returns a string representing the error.
 *
 * @return {String}
 * @api public
 */
InternalOpenIDError.prototype.toString = function toString() {
  let m = this.message;
  if (this.openidError) {
    if (this.openidError instanceof Error) {
      m += ` (${this.openidError})`;
    } else if (this.openidError.message) {
      m += ` (message: '${this.openidError.message}')`;
    }
  }
  return m;
};

/**
 * Expose `InternalOpenIDError`.
 */
module.exports = InternalOpenIDError;
